import { JwtHeader, JwtPayload } from "./jwt-model.js";
/**
 * Sanity check, decompose and JSON parse a JWT string into its constituent, and yet unverified, parts:
 * - header object
 * - payload object
 * - signature string
 *
 * This function does NOT verify a JWT, do not trust the returned payload and header!
 *
 * For most use cases, you would not want to call this function directly yourself, rather you
 * would call verify() with the JWT, which would call this function (and others) for you.
 *
 * @param jwt The JWT (as string)
 * @returns the decomposed, and yet unverified, JWT
 */
export declare function decomposeUnverifiedJwt(jwt: unknown): DecomposedJwt;
export interface DecomposedJwt {
    /**
     * The yet unverified (!) header of the JWT
     */
    header: JwtHeader;
    /**
     * The yet unverified (!) header of the JWT, as base64url-encoded string
     */
    headerB64: string;
    /**
     * The yet unverified (!) payload of the JWT
     */
    payload: JwtPayload;
    /**
     * The yet unverified (!) payload of the JWT, as base64url-encoded string
     */
    payloadB64: string;
    /**
     * The yet unverified (!) signature of the JWT, as base64url-encoded string
     */
    signatureB64: string;
}
/**
 * Validate JWT payload fields. Throws an error in case there's any validation issue.
 *
 * @param payload The (JSON parsed) JWT payload
 * @param options The options to use during validation
 * @returns void
 */
export declare function validateJwtFields(payload: JwtPayload, options: {
    issuer?: string | string[] | null;
    audience?: string | string[] | null;
    scope?: string | string[] | null;
    graceSeconds?: number;
}): void;
